#!/bin/bash
#
# Gregory Whitlock

if [[ "$EUID" -ne 0 ]]; then
	echo "Sorry, you need to run this as root"
	exit
fi

if [[ ! -e /dev/net/tun ]]; then
	echo "The TUN device is not available
You need to enable TUN before running this script"
	exit
fi

#Update Fedora
echo "We're Updating Fedora"
dnf -y update

#Installing OpenVPN and Easy-RSA
echo "We're Installing OpenVPN and Easy-RSA"
dnf -y install openvpn easy-rsa

#Creating Easy-RSA Directory
echo "Creating Easy-RSA directory"
mkdir /etc/openvpn/easy-rsa
cp -air /usr/share/easy-rsa/3/* /etc/openvpn/easy-rsa

#Initialize PKI
cd /etc/openvpn/easy-rsa
./easyrsa init-pki

#Building CA
./easyrsa --batch build-ca nopass

#Generate Diffie-Hellman
./easyrsa gen-dh

#Build Server
EASYRSA_CERT_EXPIRE=3650 ./easyrsa build-server-full server nopass

#Generate CRL
EASYRSA_CRL_DAYS=3650 ./easyrsa gen-crl

#Generate TLS key
openvpn --genkey --secret pki/ta.key

# Move the stuff we need
cp /etc/openvpn/easy-rsa/pki/ca.crt /etc/openvpn/server
cp /etc/openvpn/easy-rsa/pki/private/ca.key /etc/openvpn/server
cp /etc/openvpn/easy-rsa/pki/issued/server.crt /etc/openvpn/server
cp /etc/openvpn/easy-rsa/pki/private/server.key /etc/openvpn/server
cp /etc/openvpn/easy-rsa/pki/crl.pem  /etc/openvpn/server
cp /etc/openvpn/easy-rsa/pki/ta.key /etc/openvpn/server
cp /etc/openvpn/easy-rsa/pki/dh.pem /etc/openvpn/server

#Make OpenVPN status log directory
mkdir /var/log/openvpn

# Generate server.conf
echo port 1194 >> /etc/openvpn/server/server.conf
echo proto udp >> /etc/openvpn/server/server.conf
echo dev tun >> /etc/openvpn/server/server.conf
echo ca ca.crt >> /etc/openvpn/server/server.conf
echo cert server.crt >> /etc/openvpn/server/server.conf
echo key server.key >> /etc/openvpn/server/server.conf
echo dh dh.pem >> /etc/openvpn/server/server.conf
echo tls-auth ta.key 0 >> /etc/openvpn/server/server.conf
echo topology subnet >> /etc/openvpn/server/server.conf
echo server 10.8.0.0 255.255.255.0 >> /etc/openvpn/server/server.conf
echo ifconfig-pool-persist ipp.txt >> /etc/openvpn/server/server.conf
echo status /var/log/openvpn/openvpn-status.log >> /etc/openvpn/server/server.conf

#IP Forwarding
echo "net.ipv4.ip_forward = 1" >> /etc/sysctl.conf
sysctl --system

#Firewall Updates
firewall-cmd --add-port=1194/udp --permanent
firewall-cmd --add-masquerade --permanent
firewall-cmd --reload

#Updating SELINUX
dnf -y install policycoreutils-python
semanage port -a -t openvpn_port_t -p udp 1194
sed -i -e 's/SELINUX=enforcing/SELINUX=permissive/g' /etc/selinux/config
#Start and Enable OpenVPN Server
systemctl start openvpn-server@server.service
systemctl enable openvpn-server@server.service

#Make CCD directory
mkdir /var/ccd

#Rebooting
echo "Rebooting after install for SELINUX Changes"
shutdown -r +1
